<?php
/*
Plugin Name: WooCommerce Verifone payment gateway
Plugin URI: http://www.verifone.fi/fi/Finland/Start/
Description: Verifone payment gateway for WooCommerce.
Version: 1.1
Author: Lamia
Author URI: https://lamia.fi
*/

add_action('plugins_loaded', 'woocommerce_verifone_init', 0);
function woocommerce_verifone_init(){
	if(!class_exists('WC_Payment_Gateway')) return;

	/**
	 * localization
	 */
	load_plugin_textdomain('wc-verifone', false, dirname( plugin_basename( __FILE__ ) ) . '/languages');

	class WC_Verifone_Payment extends WC_Payment_Gateway{

		/**
		 * Constructor for the gateway.
		 */
		public function __construct(){
			$this->id            = 'verifone';
			$this->has_fields    = false;
			$this->medthod_title = 'Verifone';

			$this->icon		 	 = plugins_url( 'images/logo.png', __FILE__ );

			$this->init_form_fields();
			$this->init_settings();

			// Define user set variables
			$this->title                    = $this->get_option('title');
			$this->description              = $this->get_option('description');
			$this->testmode                 = $this->get_option('testmode');
			$this->debug		            = $this->get_option('debug');
			$this->merchant_id              = $this->get_option('merchant_id');
			$this->shop_receipt_url         = WC()->api_request_url( 'WC_Verifone_Payment' );
			$this->shop_cancel_url          = WC()->api_request_url( 'WC_Verifone_Payment' );
			$this->shop_private_key_file    = $this->get_option('shop_private_key_file');
			$this->pay_page_public_key_file = $this->get_option('pay_page_public_key_file');
			$this->language_locale          = $this->get_option('language_locale');
			$this->redirect_page_id         = $this->get_option('redirect_page_id');
			$this->liveurl_1                = $this->get_option('liveurl_1');
			$this->liveurl_2                = $this->get_option('liveurl_2');
			$this->liveurl_3                = $this->get_option('liveurl_3');
			$this->testurl                  = $this->get_option('testurl');
			$this->validate_url             = $this->get_option('validate_url');
			$this->skip_confirmation_page   = $this->get_option('skip_confirmation_page');

			// Logs
			if ( 'yes' == $this->debug ) {
				$this->log = new WC_Logger();
			}

			$this->msg['message'] = "";
			$this->msg['class'] = "";

			add_action('init', array(&$this, 'check_verifone_response'));

			// Payment listener/API hook
			add_action( 'woocommerce_api_wc_verifone_payment', array( &$this, 'check_verifone_response' ) );

			if ( version_compare( WOOCOMMERCE_VERSION, '2.0.0', '>=' ) ) {
				add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( &$this, 'process_admin_options' ) );
			} else {
				add_action( 'woocommerce_update_options_payment_gateways', array( &$this, 'process_admin_options' ) );
			}

			add_action('woocommerce_receipt_verifone', array(&$this, 'receipt_page'));

			/**
			 * Check settings
			 */
			add_action( 'admin_notices', array( $this,  'verifone_setting_check' ) );
		}

		function init_form_fields(){

			$this->form_fields = array(
				'enabled' => array(
					'title' => __('Enabled / Disabled', 'wc-verifone'),
					'type' => 'checkbox',
					'label' => __('Enable the Verifone payment module.', 'wc-verifone'),
					'default' => 'no'
				),
				'testmode' => array(
					'title'       => __( 'Test environment', 'wc-verifone' ),
					'type'        => 'checkbox',
					'label'       => __( 'Enable test mode.', 'wc-verifone' ),
					'default'     => 'no',
					'description' => __( 'The Verifone test environment can be used to test payments.', 'wc-verifone' ),
				),
				'debug' => array(
					'title'       => __( 'Debug Log', 'wc-verifone' ),
					'type'        => 'checkbox',
					'label'       => __( 'Enable logging', 'wc-verifone' ),
					'default'     => 'no',
					'description' =>  __( 'Log Verifone events, such as API requests.', 'wc-verifone' ),
				),
				'title' => array(
					'title' => __('Title', 'wc-verifone'),
					'type'=> 'text',
					'description' => __('This controls the title which the user sees during checkout.', 'wc-verifone'),
					'default' => __('Verifone', 'wc-verifone')
				),
				'description' => array(
					'title' => __('Description', 'wc-verifone'),
					'type' => 'textarea',
					'description' => __('This controls the description which the user sees during checkout.', 'wc-verifone'),
					'default' => __('Pay securely by credit or debit card or internet banking through Verifone\'s secure servers.', 'wc-verifone')
				),
				'merchant_id' => array(
					'title' => __('Merchant ID', 'wc-verifone'),
					'type' => 'text',
					'description' => __('Your unique merchant identifier.', 'wc-verifone'),
					'default' => __('demo-merchant-agreement', 'wc-verifone')
				),
				'shop_private_key_file' => array(
					'title' => __('Shop private key file', 'wc-verifone'),
					'type' => 'text',
					'description' => __('Path to private key file.', 'wc-verifone'),
					'default' => __('keys/shop-private-key.pem', 'wc-verifone')
				),
				'pay_page_public_key_file' => array(
					'title' => __('Pay page key file', 'wc-verifone'),
					'type' => 'text',
					'description' => __('Path to pay page public key file.', 'wc-verifone'),
					'default' => __('keys/pay-page-public-key.pem', 'wc-verifone')
				),
				'language_locale' => array(
					'title' => __('Language', 'wc-verifone'),
					'type' => 'select',
					'options' => $this->get_languages(),
					'description' => __('Supported language locale used by buyer.', 'wc-verifone'),
				),
				'redirect_page_id' => array(
					'title' => __('Return page', 'wc-verifone'),
					'type' => 'select',
					'options' => $this->get_pages('Select Page'),
					'description' => __('URL of success page.', 'wc-verifone'),
				),
				'testurl' => array(
					'title' => __('Test payment URL', 'wc-verifone'),
					'type' => 'text',
					'description' => __('URL to Verifone\'s test shop environment.', 'wc-verifone'),
					'default' => __('https://epayment.test.point.fi/pw/payment')
				),
				'liveurl_1' => array(
					'title' => __('URL to the payment system. Required.', 'wc-verifone'),
					'type' => 'text',
					'description' => __('The first URL used on a production environment.', 'wc-verifone'),
					'default' => __('https://epayment1.point.fi/pw/payment'),
					'custom_attributes' => array('required'=>'required')
				),
				'liveurl_2' => array(
					'title' => __('Second URL to the payment system. Required.', 'wc-verifone'),
					'type' => 'text',
					'description' => __('The second URL used on a production environment.', 'wc-verifone'),
					'default' => __('https://epayment2.point.fi/pw/payment'),
					'custom_attributes' => array('required'=>'required')
				),
				'liveurl_3' => array(
					'title' => __('Third URL to the payment system. Required.', 'wc-verifone'),
					'type' => 'text',
					'description' => __('The third URL used on a production environment.', 'wc-verifone'),
					'default' => __('https://epayment3.point.fi/pw/payment'),
					'custom_attributes' => array('required'=>'required')
				),
				'validate_url' => array(
					'title'       => __( 'Node availability', 'wc-verifone' ),
					'type'        => 'checkbox',
					'label'       => __( 'Check payment node availability.', 'wc-verifone' ),
					'default'     => 'yes',
					'description' => __( 'Make a check that payment node is available.', 'wc-verifone' ),
				) ,
				'skip_confirmation_page' => array(
					'title'       => __( 'Skip confirmation page', 'wc-verifone' ),
					'type'        => 'checkbox',
					'label'       => __( 'Enable auto return', 'wc-verifone' ),
					'default'     => 'yes',
					'description' => __( 'Return directly to shop after payment.', 'wc-verifone' ),
				)
			);
		}

		public function verifone_setting_check(){
			if( !$this->liveurl_1 )
			{
				echo "<div class=\"error\"><p>Verifone ".__('Setting', 'wc-verifone')." - ".__('URL to the payment system. Required.', 'wc-verifone')."</p></div>";
			}
			else if( !$this->liveurl_2 )
			{
				echo "<div class=\"error\"><p>Verifone ".__('Setting', 'wc-verifone')." - ".__('Second URL to the payment system. Required.', 'wc-verifone')."</p></div>";
			}
			else if( !$this->liveurl_3 )
			{
				echo "<div class=\"error\"><p>Verifone ".__('Setting', 'wc-verifone')." - ".__('Third URL to the payment system. Required.', 'wc-verifone')."</p></div>";
			}
		}

		/**
		 * Admin Panel Options
		 */
		public function admin_options(){
			echo '<h3>'.__('Verifone payment gateway', 'wc-verifone').'</h3>';
			echo '<p>'.__('Verifone is a secure way to pay online.', 'wc-verifone').'</p>';
			echo '<table class="form-table">';
			// Generate the HTML For the settings form.
			$this->generate_settings_html();
			echo '</table>';
		}

		/**
		 *  There are no payment fields for Verifone, but we want to show the description if set.
		 **/
		function payment_fields(){
			if($this -> description) echo wpautop(wptexturize($this -> description));
		}

		/**
		 * Receipt Page
		 **/
		function receipt_page($order){
			echo '<p>'.__('Thank you for your order, please click the button below to pay with Verifone.', 'wc-verifone').'</p>';
			echo $this -> generate_verifone_form($order);
		}

		/**
		 * Generate Verifone button link
		 **/
		public function generate_verifone_form($order_id){

			global $woocommerce;
			$order = new WC_Order( $order_id );
			$txnid = $order_id.'_'.date("ymds");

			$redirect_url = ($this -> redirect_page_id=="" || $this -> redirect_page_id==0)?get_site_url() . "/":get_permalink($this -> redirect_page_id);

			if ( 'yes' == $this->testmode ) {
				$verifone_adr = $this->testurl;
			} else {
				$verifone_adr = $this->get_url();
			}

			$verifone_args = $this->get_verifone_args( $order );

			$verifone_args_array = array();
			foreach($verifone_args as $key => $value){
				$verifone_args_array[] = "<input type='hidden' name='$key' value='$value'/>";
			}
			return '<form action="'.$verifone_adr.'" method="post" id="verifone_form">
            ' . implode('', $verifone_args_array) . '
            <input type="submit" class="button-alt" id="submit_verifone_form" value="'.__('Pay via Verifone', 'wc-verifone').'" /> <a class="button cancel" href="'.$order->get_cancel_order_url().'">'.__('Cancel order and restore cart', 'wc-verifone').'</a>
            <script type="text/javascript">
jQuery(function(){
jQuery("body").block(
        {
            message: "<img src=\"'.$woocommerce->plugin_url().'/assets/images/ajax-loader.gif\" alt=\"Redirecting…\" style=\"float:left; margin-right: 10px;\" />'.__('Thank you for your order. We are now redirecting you to the payment gateway to make your payment.', 'wc-verifone').'",
                overlayCSS:
        {
            background: "#fff",
                opacity: 0.6
    },
    css: {
        padding:        20,
            textAlign:      "center",
            color:          "#555",
            border:         "3px solid #aaa",
            backgroundColor:"#fff",
            cursor:         "wait",
            lineHeight:"32px"
    }
    });
    jQuery("#submit_verifone_form").click();
	});</script>
            </form>';
		}

		/**
		 * Get Verifone Args
		 *
		 * @param WC_Order $order
		 * @return array
		 */
		function get_verifone_args( $order ) {

			$order_id = $order->id;

			if ( 'yes' == $this->debug )
			{
				$this->log->add( 'verifone', get_bloginfo( 'name' ).' - Generating payment form for order ' . $order->get_order_number() . '. Notify URL: ' . $this->shop_receipt_url );
			}

			//date_default_timezone_set('Europe/Helsinki');
			date_default_timezone_set('UTC');
			$orderdate     = date("Y-m-d H:i:s");
			$micotime      = sprintf("%.0f",microtime(true) * 1000000);
			$tokendata     = sprintf("%s;%s;%s", $this->merchant_id, $order->id, $orderdate );
			$order_country = $order->billing_country;

			$order_vat_amount   = $order->get_total_tax();
			$order_gross_amount = $order->get_total();
			$order_net_amount   = $order->get_subtotal();

			$order_vat_perc = round($order_vat_amount * 100 / $order_net_amount, 2);

			//Add shipping to net total
			$order_net_amount   += $order->get_total_shipping();

			$order_vat_perc     = $this->formatApiNumber($order_vat_perc);
			$order_gross_amount = $this->formatApiNumber($order_gross_amount);
			$order_net_amount   = $this->formatApiNumber($order_net_amount);
			$order_vat_amount   = $this->formatApiNumber($order_vat_amount);

			$verifone_args['i-f-1-11_interface-version'] = "3";
			$verifone_args['i-f-1-3_order-currency-code'] = $this->get_currency_number();
			$verifone_args['i-t-1-3_delivery-address-country-code'] = $this->get_countries($order_country);
			$verifone_args['i-t-1-4_order-vat-percentage'] = $order_vat_perc;
			$verifone_args['l-f-1-20_order-gross-amount'] = $order_gross_amount;
			$verifone_args['l-f-1-20_order-net-amount'] = $order_net_amount;
			$verifone_args['l-f-1-20_order-vat-amount'] = $order_vat_amount;

			$verifone_args['locale-f-2-5_payment-locale'] = $this->language_locale;
			$verifone_args['s-f-1-100_buyer-email-address'] = $this->verifone_item_email($order->billing_email);
			$verifone_args['s-f-1-10_software-version'] = "1.0.1";
			$verifone_args['s-f-1-30_buyer-first-name'] = $this->verifone_item_length($order->billing_first_name);
			$verifone_args['s-f-1-30_buyer-last-name'] = $this->verifone_item_length($order->billing_last_name);
			$verifone_args['s-f-1-30_software'] = "Woocommerce";
			$verifone_args['s-f-1-36_merchant-agreement-code'] = $this->merchant_id;
			$verifone_args['s-f-1-36_order-number'] = $order->id;

			$verifone_args['s-f-32-32_payment-token'] = strtoupper(substr(hash('sha256',$tokendata),0,32));

			$verifone_args['s-f-5-128_cancel-url']   = $this->shop_receipt_url;
			$verifone_args['s-f-5-128_error-url']    = $this->shop_receipt_url;
			$verifone_args['s-f-5-128_expired-url']  = $this->shop_receipt_url;
			$verifone_args['s-f-5-128_rejected-url'] = $this->shop_receipt_url;
			$verifone_args['s-f-5-128_success-url']  = $this->shop_receipt_url;
			$verifone_args['s-t-1-30_buyer-phone-number'] = "";
			$verifone_args['s-t-1-30_delivery-address-city']        = $this->verifone_item_length($order->billing_city);
			$verifone_args['s-t-1-30_delivery-address-line-one']    = $this->verifone_item_length($order->billing_address_1);
			$verifone_args['s-t-1-30_delivery-address-line-three']  = $this->verifone_item_length($order->billing_state);
			$verifone_args['s-t-1-30_delivery-address-line-two']    = $this->verifone_item_length($order->billing_address_2);
			$verifone_args['s-t-1-30_delivery-address-postal-code'] = $this->verifone_item_length($order->billing_postcode);

			$verifone_args['s-t-1-36_order-note'] = "Note for payment";
			$verifone_args['t-f-14-19_order-timestamp'] = $orderdate;
			$verifone_args['t-f-14-19_payment-timestamp'] = $orderdate;

			$item_loop = 0;

			// Items
			if ( sizeof( $order->get_items() ) > 0 ) {
				foreach ( $order->get_items() as $item ) {
					if ( ! $item['qty'] ) {
						continue;
					}

					$product   = $order->get_product_from_item( $item );
					$item_name = $item['name'];
					$item_meta = new WC_Order_Item_Meta( $item['item_meta'] );

					if ( $meta = $item_meta->display( true, true ) ) {
						$item_name .= ' ( ' . $meta . ' )';
					}

					$item_net = round($order->get_line_subtotal( $item, false ),2);
					$item_tax = round($order->get_line_tax( $item, false ),2);

					$item_vat_perc = round($item_tax * 100 / $item_net, 2);
					$item_gross = $item_net + $item_tax;

					$item_vat_perc = $this->formatApiNumber($item_vat_perc);
					$item_gross = $this->formatApiNumber($item_gross );
					$item_net = $this->formatApiNumber($item_net  );

					$verifone_args[ 's-t-1-30_bi-name-' . $item_loop ]               = $this->verifone_item_length( $item_name );
					$verifone_args[ 'i-t-1-11_bi-unit-count-' . $item_loop ]         = $item['qty'];
					$verifone_args[ 'i-t-1-4_bi-discount-percentage-' . $item_loop ] = 0;
					$verifone_args[ 'i-t-1-4_bi-vat-percentage-' . $item_loop ]      = $item_vat_perc;
					$verifone_args[ 'l-t-1-20_bi-gross-amount-' . $item_loop ]       = $item_gross;
					$verifone_args[ 'l-t-1-20_bi-net-amount-' . $item_loop ]         = $item_net;
					$verifone_args[ 'l-t-1-20_bi-unit-cost-' . $item_loop ]          = $item_net;

					if ( $verifone_args[ 'l-t-1-20_bi-gross-amount-' . $item_loop ] < 0 ) {
						return false; // Abort - negative line
					}

					$item_loop ++;
				}
			}

			//Shipping
			if($order->get_total_shipping() != 0) {

				$item_net = round($order->get_total_shipping(),2);
				$item_tax = round($order->get_shipping_tax(),2);

				$item_vat_perc = round($item_tax * 100 / $item_net, 2);
				$item_gross    = $item_net + $item_tax;

				$item_vat_perc = $this->formatApiNumber( $item_vat_perc );
				$item_gross    = $this->formatApiNumber( $item_gross );
				$item_net      = $this->formatApiNumber( $item_net );
				$item_name     = __( 'Shipping via', 'wc-verifone' ) . ' ' . ucwords( $order->get_shipping_method() );

				$verifone_args['s-t-1-30_bi-name-' . $item_loop]               = $this->verifone_item_length( $item_name );
				$verifone_args['i-t-1-11_bi-unit-count-' . $item_loop]         = 1;
				$verifone_args['i-t-1-4_bi-discount-percentage-' . $item_loop] = 0;
				$verifone_args['i-t-1-4_bi-vat-percentage-' . $item_loop]      = $item_vat_perc;
				$verifone_args['l-t-1-20_bi-gross-amount-' . $item_loop]       = $item_gross;
				$verifone_args['l-t-1-20_bi-net-amount-' . $item_loop]         = $item_net;
				$verifone_args['l-t-1-20_bi-unit-cost-' . $item_loop]          = $item_net;
			}

			if ( 'yes' == $this->skip_confirmation_page ) {
				$verifone_args["i-t-1-1_skip-confirmation-page"] = "1";
			}

			//$verifone_args["s-t-5-256_change-server-to-server-success-url"] = "http://onet.pl";

			$datatosign = $this->formatparameters($verifone_args);
			$verifone_args["s-t-256-256_signature-one"] = $this->generatesignature($datatosign);
			return $verifone_args;
		}

		function formatparameters($parameters) {

			ksort($parameters, SORT_STRING);
			$formatedresult = "";
			foreach ($parameters as $key => $value)  {
				$value = str_replace(';',';;',$value);
				$formatedresult .= sprintf("%s=%s;", $key, $value);
			}
			return $formatedresult;
		}

		/** Creates signature for data
		 * @param $data data to creta
		 * @return string
		 */
		function generatesignature($data) {
			include( plugin_dir_path( __FILE__ ) . 'Crypt/RSA.php');

			$rsa = new Crypt_RSA();
			$rsa->setSignatureMode(CRYPT_RSA_SIGNATURE_PKCS1);
			$rsa->loadKey(file_get_contents($this->shop_private_key_file, true));
			$signature = $rsa->sign($data);

			return strtoupper(bin2hex($signature));
		}

		/** Verifies the data signature
		 * @param $datatoverify data to verify signature for
		 * @param $signaturedata signature data as hex string
		 * @return bool true - signature valid, false - invalid
		 */
		function verifysignature($datatoverify, $signaturedata) {
			include( plugin_dir_path( __FILE__ ) . 'Crypt/RSA.php');
			$rsa = new Crypt_RSA();
			$rsa->setSignatureMode(CRYPT_RSA_SIGNATURE_PKCS1);
			$rsa->loadKey(file_get_contents($this->pay_page_public_key_file, true));
			return $rsa->verify($datatoverify, pack("H*" , $signaturedata)) ? true : false;
		}

		/**
		 * Process the payment and return the result
		 */
		function process_payment($order_id)
		{
			$order = new WC_Order($order_id);
			return array(
				'result' 	=> 'success',
				'redirect'	=> $order->get_checkout_payment_url( true )
			);
		}

		/**
		 * Check for valid Verifone server callback
		 **/
		function check_verifone_response(){

			global $woocommerce;
			if(isset($_POST['s-f-1-36_order-number'])){

				if ( 'yes' == $this->debug ) {
					$this->log->add( 'verifone', __('Api response for order', 'wc-verifone').' - '.$_POST['s-f-1-36_order-number']  );
				}

				$order = new WC_Order( (int)$_POST['s-f-1-36_order-number'] );

				unset($_POST["s-t-1-40_shop-order__phase"]);
				unset($_POST["s-t-1-40_shop-receipt__phase"]);
				$signedformdata = $_POST;
				$renderdata = $_POST;
				unset($signedformdata["s-t-256-256_signature-one"]);
				unset($signedformdata["s-t-256-256_signature-two"]);
				print_r($signedformdata);
				$signature_valid = $this->verifysignature($this->formatparameters($signedformdata), $_POST["s-t-256-256_signature-one"]);

				if(!$signature_valid) {
					$this->msg['class'] = 'error';
					$this->msg['message'] = __('Security Error. Illegal access detected', 'wc-verifone');
				}
				elseif(isset($_POST['s-t-1-30_cancel-reason'])) {
					$this->msg['class'] = 'error';
					$message_error = $this->get_error_message($_POST['s-t-1-30_cancel-reason']);
					$this->msg['message'] = __($message_error, 'wc-verifone');
				}
				elseif($signature_valid and isset($_POST['s-f-1-36_order-number']) and !empty($_POST['s-f-1-36_order-number']) and !isset($_POST['s-t-1-30_cancel-reason'])) {

					// the order might be handled already
					if($order->status !=='completed'){

						//Set order as paid
						$order->payment_complete();

						$this->msg['class'] = 'success';
						$this->msg['message'] = __('Payment received', 'wc-verifone');

						$order->add_order_note(__('Verifone payment successful', 'wc-verifone').'<br/>Ref Number/Transaction ID: '.preg_replace("/[^0-9]+/","",$_POST['l-f-1-20_transaction-number']));
						$order->add_order_note($this->msg['message']);

						$order->update_status( 'completed' );

						// Empty the Cart
						WC()->cart->empty_cart();
					}
					else
					{
						$this->msg['class'] = 'notice';
						$this->msg['message'] = __('Payment received', 'wc-verifone');
					}
				}
				else
				{
					$this->msg['class'] = 'error';
					$this->msg['message'] = __('Something went wrong', 'wc-verifone');
				}

				//Log response message
				if ( 'yes' == $this->debug ) {
					$this->log->add( 'verifone', __('Api response for order', 'wc-verifone').' - '.$_POST['s-f-1-36_order-number'] .' - '.$_POST['s-t-1-30_cancel-reason'].' - '.$this->msg['class']. ' - '. $this->msg['message'] );
				}

				// Get redirect
				if($this->msg['class'] == 'success')
				{
					$return_url = $order->get_checkout_order_received_url();

					//If redirect success is set
					if($this->redirect_page_id){
						$return_url = get_permalink($this -> redirect_page_id)  ;
					}
				}
				else if($this->msg['class'] == 'notice')
				{
					$order->add_order_note($this->msg['message']);

					$return_url = $order->get_checkout_order_received_url();
				}
				else
				{
					$order->update_status('failed');
					$order->add_order_note('Failed');
					$order->add_order_note($this->msg['message']);

					$return_url = $order->get_cancel_order_url();
				}

				wc_add_notice( $this->msg['message'], $this->msg['class']);

				// Redirect to success/confirmation/payment page
				wp_safe_redirect(
					apply_filters( 'woocommerce_checkout_no_payment_needed_redirect', $return_url, $order )
				);
			}
		}

		/**
		 * Limit the length of item names to 30 !
		 * @param  string $item_name
		 * @return string
		 */
		public function verifone_item_length( $item_name ) {
			if ( strlen( $item_name ) > 30 ) {
				$item_name = substr( $item_name, 0, 27 ) . '...';
			}
			return html_entity_decode( $item_name, ENT_NOQUOTES, 'UTF-8' );
		}

		/**
		 * Limit the length of email to 100 !
		 * @param  string $item_email
		 * @return string
		 */
		public function verifone_item_email( $item_email ) {
			if ( strlen( $item_email ) > 100 ) {
				$item_email = substr( $item_email, 0, 97 ) . '...';
			}
			return html_entity_decode( $item_email, ENT_NOQUOTES, 'UTF-8' );
		}

		/**
		 * Format number to Api Format 23,5EUR => 2350
		 */
		public function formatApiNumber( $num_value ) {

			return number_format((float)$num_value, 2, '', '');
		}

		/**
		 * Get Supported locales by Verifone
		 */
		function get_languages() {

			$page_languages['fi_FI'] = 'FI';
			$page_languages['sv_SE'] = 'SE';
			$page_languages['no_NO'] = 'NO';
			$page_languages['en_GB'] = 'GB';

			return $page_languages;
		}

		public function get_url() {

			$payment_urls   = array();
			$payment_urls[] = $this->liveurl_1;
			$second_url     = $this->liveurl_2;
			$third_url      = $this->liveurl_3;
			if(!empty($second_url)) { $payment_urls[] = $second_url; }
			if(!empty($third_url))  { $payment_urls[] = $third_url; }

			shuffle($payment_urls);

			foreach($payment_urls as $url) {
				if($this->validate_url == 'no') {
					return $url;
				}

				// GET must result HTTP 200 OK and empty contents
				// anything else indicates that the payment system node is not available
				// file_get_contents requires valid ssl certificate
				$fc = @file_get_contents($url);
				if($fc !== false and $fc == ''
					and strpos($http_response_header[0],'200 OK') !== false) {
					return $url;
				}
			}
			// flag error
			return false;
		}
		/**
		 * Get Supported currency number by ISO 4217
		 */
		function get_currency_number(){

			$shop_currency = get_option('woocommerce_currency');

			// http://en.wikipedia.org/wiki/ISO_4217
			$currency_4217 = array(
				'AFA' => array('Afghan Afghani', '971'),
				'AWG' => array('Aruban Florin', '533'),
				'AUD' => array('Australian Dollars', '036'),
				'ARS' => array('Argentine Pes', '032'),
				'AZN' => array('Azerbaijanian Manat', '944'),
				'BSD' => array('Bahamian Dollar', '044'),
				'BDT' => array('Bangladeshi Taka', '050'),
				'BBD' => array('Barbados Dollar', '052'),
				'BYR' => array('Belarussian Rouble', '974'),
				'BOB' => array('Bolivian Boliviano', '068'),
				'BRL' => array('Brazilian Real', '986'),
				'GBP' => array('British Pounds Sterling', '826'),
				'BGN' => array('Bulgarian Lev', '975'),
				'KHR' => array('Cambodia Riel', '116'),
				'CAD' => array('Canadian Dollars', '124'),
				'KYD' => array('Cayman Islands Dollar', '136'),
				'CLP' => array('Chilean Peso', '152'),
				'CNY' => array('Chinese Renminbi Yuan', '156'),
				'COP' => array('Colombian Peso', '170'),
				'CRC' => array('Costa Rican Colon', '188'),
				'HRK' => array('Croatia Kuna', '191'),
				'CPY' => array('Cypriot Pounds', '196'),
				'CZK' => array('Czech Koruna', '203'),
				'DKK' => array('Danish Krone', '208'),
				'DOP' => array('Dominican Republic Peso', '214'),
				'XCD' => array('East Caribbean Dollar', '951'),
				'EGP' => array('Egyptian Pound', '818'),
				'ERN' => array('Eritrean Nakfa', '232'),
				'EEK' => array('Estonia Kroon', '233'),
				'EUR' => array('Euro', '978'),
				'GEL' => array('Georgian Lari', '981'),
				'GHC' => array('Ghana Cedi', '288'),
				'GIP' => array('Gibraltar Pound', '292'),
				'GTQ' => array('Guatemala Quetzal', '320'),
				'HNL' => array('Honduras Lempira', '340'),
				'HKD' => array('Hong Kong Dollars', '344'),
				'HUF' => array('Hungary Forint', '348'),
				'ISK' => array('Icelandic Krona', '352'),
				'INR' => array('Indian Rupee', '356'),
				'IDR' => array('Indonesia Rupiah', '360'),
				'ILS' => array('Israel Shekel', '376'),
				'JMD' => array('Jamaican Dollar', '388'),
				'JPY' => array('Japanese yen', '392'),
				'KZT' => array('Kazakhstan Tenge', '368'),
				'KES' => array('Kenyan Shilling', '404'),
				'KWD' => array('Kuwaiti Dinar', '414'),
				'LVL' => array('Latvia Lat', '428'),
				'LBP' => array('Lebanese Pound', '422'),
				'LTL' => array('Lithuania Litas', '440'),
				'MOP' => array('Macau Pataca', '446'),
				'MKD' => array('Macedonian Denar', '807'),
				'MGA' => array('Malagascy Ariary', '969'),
				'MYR' => array('Malaysian Ringgit', '458'),
				'MTL' => array('Maltese Lira', '470'),
				'BAM' => array('Marka', '977'),
				'MUR' => array('Mauritius Rupee', '480'),
				'MXN' => array('Mexican Pesos', '484'),
				'MZM' => array('Mozambique Metical', '508'),
				'NPR' => array('Nepalese Rupee', '524'),
				'ANG' => array('Netherlands Antilles Guilder', '532'),
				'TWD' => array('New Taiwanese Dollars', '901'),
				'NZD' => array('New Zealand Dollars', '554'),
				'NIO' => array('Nicaragua Cordoba', '558'),
				'NGN' => array('Nigeria Naira', '566'),
				'KPW' => array('North Korean Won', '408'),
				'NOK' => array('Norwegian Krone', '578'),
				'OMR' => array('Omani Riyal', '512'),
				'PKR' => array('Pakistani Rupee', '586'),
				'PYG' => array('Paraguay Guarani', '600'),
				'PEN' => array('Peru New Sol', '604'),
				'PHP' => array('Philippine Pesos', '608'),
				'PLN' => array('Polish złoty', '985'),
				'QAR' => array('Qatari Riyal', '634'),
				'RON' => array('Romanian New Leu', '946'),
				'RUB' => array('Russian Federation Ruble', '643'),
				'SAR' => array('Saudi Riyal', '682'),
				'CSD' => array('Serbian Dinar', '891'),
				'SCR' => array('Seychelles Rupee', '690'),
				'SGD' => array('Singapore Dollars', '702'),
				'SKK' => array('Slovak Koruna', '703'),
				'SIT' => array('Slovenia Tolar', '705'),
				'ZAR' => array('South African Rand', '710'),
				'KRW' => array('South Korean Won', '410'),
				'LKR' => array('Sri Lankan Rupee', '144'),
				'SRD' => array('Surinam Dollar', '968'),
				'SEK' => array('Swedish Krona', '752'),
				'CHF' => array('Swiss Francs', '756'),
				'TZS' => array('Tanzanian Shilling', '834'),
				'THB' => array('Thai Baht', '764'),
				'TTD' => array('Trinidad and Tobago Dollar', '780'),
				'TRY' => array('Turkish New Lira', '949'),
				'AED' => array('UAE Dirham', '784'),
				'USD' => array('US Dollars', '840'),
				'UGX' => array('Ugandian Shilling', '800'),
				'UAH' => array('Ukraine Hryvna', '980'),
				'UYU' => array('Uruguayan Peso', '858'),
				'UZS' => array('Uzbekistani Som', '860'),
				'VEB' => array('Venezuela Bolivar', '862'),
				'VND' => array('Vietnam Dong', '704'),
				'AMK' => array('Zambian Kwacha', '894'),
				'ZWD' => array('Zimbabwe Dollar', '716'),
			);

			if(isset($currency_4217[$shop_currency][1])) {
				return $currency_4217[$shop_currency][1];
			} else {
				return "978";  // default to EUR
			}
		}

		/**
		 * Get Supported countries number by ISO 3166
		 */
		function get_countries($order_country)
		{
			$countries_3166 = array(
				'AF'=> array('AFG','004','ISO 3166-2:AF'),
				'AX'=> array('ALA','248','ISO 3166-2:AX'),
				'AL'=> array('ALB','008','ISO 3166-2:AL'),
				'DZ'=> array('DZA','012','ISO 3166-2:DZ'),
				'AS'=> array('ASM','016','ISO 3166-2:AS'),
				'AD'=> array('AND','020','ISO 3166-2:AD'),
				'AO'=> array('AGO','024','ISO 3166-2:AO'),
				'AI'=> array('AIA','660','ISO 3166-2:AI'),
				'AQ'=> array('ATA','010','ISO 3166-2:AQ'),
				'AG'=> array('ATG','028','ISO 3166-2:AG'),
				'AR'=> array('ARG','032','ISO 3166-2:AR'),
				'AM'=> array('ARM','051','ISO 3166-2:AM'),
				'AW'=> array('ABW','533','ISO 3166-2:AW'),
				'AU'=> array('AUS','036','ISO 3166-2:AU'),
				'AT'=> array('AUT','040','ISO 3166-2:AT'),
				'AZ'=> array('AZE','031','ISO 3166-2:AZ'),
				'BS'=> array('BHS','044','ISO 3166-2:BS'),
				'BH'=> array('BHR','048','ISO 3166-2:BH'),
				'BD'=> array('BGD','050','ISO 3166-2:BD'),
				'BB'=> array('BRB','052','ISO 3166-2:BB'),
				'BY'=> array('BLR','112','ISO 3166-2:BY'),
				'BE'=> array('BEL','056','ISO 3166-2:BE'),
				'BZ'=> array('BLZ','084','ISO 3166-2:BZ'),
				'BJ'=> array('BEN','204','ISO 3166-2:BJ'),
				'BM'=> array('BMU','060','ISO 3166-2:BM'),
				'BT'=> array('BTN','064','ISO 3166-2:BT'),
				'BO'=> array('BOL','068','ISO 3166-2:BO'),
				'BQ'=> array('BES','535','ISO 3166-2:BQ'),
				'BA'=> array('BIH','070','ISO 3166-2:BA'),
				'BW'=> array('BWA','072','ISO 3166-2:BW'),
				'BV'=> array('BVT','074','ISO 3166-2:BV'),
				'BR'=> array('BRA','076','ISO 3166-2:BR'),
				'IO'=> array('IOT','086','ISO 3166-2:IO'),
				'BN'=> array('BRN','096','ISO 3166-2:BN'),
				'BG'=> array('BGR','100','ISO 3166-2:BG'),
				'BF'=> array('BFA','854','ISO 3166-2:BF'),
				'BI'=> array('BDI','108','ISO 3166-2:BI'),
				'KH'=> array('KHM','116','ISO 3166-2:KH'),
				'CM'=> array('CMR','120','ISO 3166-2:CM'),
				'CA'=> array('CAN','124','ISO 3166-2:CA'),
				'CV'=> array('CPV','132','ISO 3166-2:CV'),
				'KY'=> array('CYM','136','ISO 3166-2:KY'),
				'CF'=> array('CAF','140','ISO 3166-2:CF'),
				'TD'=> array('TCD','148','ISO 3166-2:TD'),
				'CL'=> array('CHL','152','ISO 3166-2:CL'),
				'CN'=> array('CHN','156','ISO 3166-2:CN'),
				'CX'=> array('CXR','162','ISO 3166-2:CX'),
				'CC'=> array('CCK','166','ISO 3166-2:CC'),
				'CO'=> array('COL','170','ISO 3166-2:CO'),
				'KM'=> array('COM','174','ISO 3166-2:KM'),
				'CG'=> array('COG','178','ISO 3166-2:CG'),
				'CD'=> array('COD','180','ISO 3166-2:CD'),
				'CK'=> array('COK','184','ISO 3166-2:CK'),
				'CR'=> array('CRI','188','ISO 3166-2:CR'),
				'CI'=> array('CIV','384','ISO 3166-2:CI'),
				'HR'=> array('HRV','191','ISO 3166-2:HR'),
				'CU'=> array('CUB','192','ISO 3166-2:CU'),
				'CW'=> array('CUW','531','ISO 3166-2:CW'),
				'CY'=> array('CYP','196','ISO 3166-2:CY'),
				'CZ'=> array('CZE','203','ISO 3166-2:CZ'),
				'DK'=> array('DNK','208','ISO 3166-2:DK'),
				'DJ'=> array('DJI','262','ISO 3166-2:DJ'),
				'DM'=> array('DMA','212','ISO 3166-2:DM'),
				'DO'=> array('DOM','214','ISO 3166-2:DO'),
				'EC'=> array('ECU','218','ISO 3166-2:EC'),
				'EG'=> array('EGY','818','ISO 3166-2:EG'),
				'SV'=> array('SLV','222','ISO 3166-2:SV'),
				'GQ'=> array('GNQ','226','ISO 3166-2:GQ'),
				'ER'=> array('ERI','232','ISO 3166-2:ER'),
				'EE'=> array('EST','233','ISO 3166-2:EE'),
				'ET'=> array('ETH','231','ISO 3166-2:ET'),
				'FK'=> array('FLK','238','ISO 3166-2:FK'),
				'FO'=> array('FRO','234','ISO 3166-2:FO'),
				'FJ'=> array('FJI','242','ISO 3166-2:FJ'),
				'FI'=> array('FIN','246','ISO 3166-2:FI'),
				'FR'=> array('FRA','250','ISO 3166-2:FR'),
				'GF'=> array('GUF','254','ISO 3166-2:GF'),
				'PF'=> array('PYF','258','ISO 3166-2:PF'),
				'TF'=> array('ATF','260','ISO 3166-2:TF'),
				'GA'=> array('GAB','266','ISO 3166-2:GA'),
				'GM'=> array('GMB','270','ISO 3166-2:GM'),
				'GE'=> array('GEO','268','ISO 3166-2:GE'),
				'DE'=> array('DEU','276','ISO 3166-2:DE'),
				'GH'=> array('GHA','288','ISO 3166-2:GH'),
				'GI'=> array('GIB','292','ISO 3166-2:GI'),
				'GR'=> array('GRC','300','ISO 3166-2:GR'),
				'GL'=> array('GRL','304','ISO 3166-2:GL'),
				'GD'=> array('GRD','308','ISO 3166-2:GD'),
				'GP'=> array('GLP','312','ISO 3166-2:GP'),
				'GU'=> array('GUM','316','ISO 3166-2:GU'),
				'GT'=> array('GTM','320','ISO 3166-2:GT'),
				'GG'=> array('GGY','831','ISO 3166-2:GG'),
				'GN'=> array('GIN','324','ISO 3166-2:GN'),
				'GW'=> array('GNB','624','ISO 3166-2:GW'),
				'GY'=> array('GUY','328','ISO 3166-2:GY'),
				'HT'=> array('HTI','332','ISO 3166-2:HT'),
				'HM'=> array('HMD','334','ISO 3166-2:HM'),
				'VA'=> array('VAT','336','ISO 3166-2:VA'),
				'HN'=> array('HND','340','ISO 3166-2:HN'),
				'HK'=> array('HKG','344','ISO 3166-2:HK'),
				'HU'=> array('HUN','348','ISO 3166-2:HU'),
				'IS'=> array('ISL','352','ISO 3166-2:IS'),
				'IN'=> array('IND','356','ISO 3166-2:IN'),
				'ID'=> array('IDN','360','ISO 3166-2:ID'),
				'IR'=> array('IRN','364','ISO 3166-2:IR'),
				'IQ'=> array('IRQ','368','ISO 3166-2:IQ'),
				'IE'=> array('IRL','372','ISO 3166-2:IE'),
				'IM'=> array('IMN','833','ISO 3166-2:IM'),
				'IL'=> array('ISR','376','ISO 3166-2:IL'),
				'IT'=> array('ITA','380','ISO 3166-2:IT'),
				'JM'=> array('JAM','388','ISO 3166-2:JM'),
				'JP'=> array('JPN','392','ISO 3166-2:JP'),
				'JE'=> array('JEY','832','ISO 3166-2:JE'),
				'JO'=> array('JOR','400','ISO 3166-2:JO'),
				'KZ'=> array('KAZ','398','ISO 3166-2:KZ'),
				'KE'=> array('KEN','404','ISO 3166-2:KE'),
				'KI'=> array('KIR','296','ISO 3166-2:KI'),
				'KP'=> array('PRK','408','ISO 3166-2:KP'),
				'KR'=> array('KOR','410','ISO 3166-2:KR'),
				'KW'=> array('KWT','414','ISO 3166-2:KW'),
				'KG'=> array('KGZ','417','ISO 3166-2:KG'),
				'LA'=> array('LAO','418','ISO 3166-2:LA'),
				'LV'=> array('LVA','428','ISO 3166-2:LV'),
				'LB'=> array('LBN','422','ISO 3166-2:LB'),
				'LS'=> array('LSO','426','ISO 3166-2:LS'),
				'LR'=> array('LBR','430','ISO 3166-2:LR'),
				'LY'=> array('LBY','434','ISO 3166-2:LY'),
				'LI'=> array('LIE','438','ISO 3166-2:LI'),
				'LT'=> array('LTU','440','ISO 3166-2:LT'),
				'LU'=> array('LUX','442','ISO 3166-2:LU'),
				'MO'=> array('MAC','446','ISO 3166-2:MO'),
				'MK'=> array('MKD','807','ISO 3166-2:MK'),
				'MG'=> array('MDG','450','ISO 3166-2:MG'),
				'MW'=> array('MWI','454','ISO 3166-2:MW'),
				'MY'=> array('MYS','458','ISO 3166-2:MY'),
				'MV'=> array('MDV','462','ISO 3166-2:MV'),
				'ML'=> array('MLI','466','ISO 3166-2:ML'),
				'MT'=> array('MLT','470','ISO 3166-2:MT'),
				'MH'=> array('MHL','584','ISO 3166-2:MH'),
				'MQ'=> array('MTQ','474','ISO 3166-2:MQ'),
				'MR'=> array('MRT','478','ISO 3166-2:MR'),
				'MU'=> array('MUS','480','ISO 3166-2:MU'),
				'YT'=> array('MYT','175','ISO 3166-2:YT'),
				'MX'=> array('MEX','484','ISO 3166-2:MX'),
				'FM'=> array('FSM','583','ISO 3166-2:FM'),
				'MD'=> array('MDA','498','ISO 3166-2:MD'),
				'MC'=> array('MCO','492','ISO 3166-2:MC'),
				'MN'=> array('MNG','496','ISO 3166-2:MN'),
				'ME'=> array('MNE','499','ISO 3166-2:ME'),
				'MS'=> array('MSR','500','ISO 3166-2:MS'),
				'MA'=> array('MAR','504','ISO 3166-2:MA'),
				'MZ'=> array('MOZ','508','ISO 3166-2:MZ'),
				'MM'=> array('MMR','104','ISO 3166-2:MM'),
				'NA'=> array('NAM','516','ISO 3166-2:NA'),
				'NR'=> array('NRU','520','ISO 3166-2:NR'),
				'NP'=> array('NPL','524','ISO 3166-2:NP'),
				'NL'=> array('NLD','528','ISO 3166-2:NL'),
				'NC'=> array('NCL','540','ISO 3166-2:NC'),
				'NZ'=> array('NZL','554','ISO 3166-2:NZ'),
				'NI'=> array('NIC','558','ISO 3166-2:NI'),
				'NE'=> array('NER','562','ISO 3166-2:NE'),
				'NG'=> array('NGA','566','ISO 3166-2:NG'),
				'NU'=> array('NIU','570','ISO 3166-2:NU'),
				'NF'=> array('NFK','574','ISO 3166-2:NF'),
				'MP'=> array('MNP','580','ISO 3166-2:MP'),
				'NO'=> array('NOR','578','ISO 3166-2:NO'),
				'OM'=> array('OMN','512','ISO 3166-2:OM'),
				'PK'=> array('PAK','586','ISO 3166-2:PK'),
				'PW'=> array('PLW','585','ISO 3166-2:PW'),
				'PS'=> array('PSE','275','ISO 3166-2:PS'),
				'PA'=> array('PAN','591','ISO 3166-2:PA'),
				'PG'=> array('PNG','598','ISO 3166-2:PG'),
				'PY'=> array('PRY','600','ISO 3166-2:PY'),
				'PE'=> array('PER','604','ISO 3166-2:PE'),
				'PH'=> array('PHL','608','ISO 3166-2:PH'),
				'PN'=> array('PCN','612','ISO 3166-2:PN'),
				'PL'=> array('POL','616','ISO 3166-2:PL'),
				'PT'=> array('PRT','620','ISO 3166-2:PT'),
				'PR'=> array('PRI','630','ISO 3166-2:PR'),
				'QA'=> array('QAT','634','ISO 3166-2:QA'),
				'RE'=> array('REU','638','ISO 3166-2:RE'),
				'RO'=> array('ROU','642','ISO 3166-2:RO'),
				'RU'=> array('RUS','643','ISO 3166-2:RU'),
				'RW'=> array('RWA','646','ISO 3166-2:RW'),
				'BL'=> array('BLM','652','ISO 3166-2:BL'),
				'SH'=> array('SHN','654','ISO 3166-2:SH'),
				'KN'=> array('KNA','659','ISO 3166-2:KN'),
				'LC'=> array('LCA','662','ISO 3166-2:LC'),
				'MF'=> array('MAF','663','ISO 3166-2:MF'),
				'PM'=> array('SPM','666','ISO 3166-2:PM'),
				'VC'=> array('VCT','670','ISO 3166-2:VC'),
				'WS'=> array('WSM','882','ISO 3166-2:WS'),
				'SM'=> array('SMR','674','ISO 3166-2:SM'),
				'ST'=> array('STP','678','ISO 3166-2:ST'),
				'SA'=> array('SAU','682','ISO 3166-2:SA'),
				'SN'=> array('SEN','686','ISO 3166-2:SN'),
				'RS'=> array('SRB','688','ISO 3166-2:RS'),
				'SC'=> array('SYC','690','ISO 3166-2:SC'),
				'SL'=> array('SLE','694','ISO 3166-2:SL'),
				'SG'=> array('SGP','702','ISO 3166-2:SG'),
				'SX'=> array('SXM','534','ISO 3166-2:SX'),
				'SK'=> array('SVK','703','ISO 3166-2:SK'),
				'SI'=> array('SVN','705','ISO 3166-2:SI'),
				'SB'=> array('SLB','090','ISO 3166-2:SB'),
				'SO'=> array('SOM','706','ISO 3166-2:SO'),
				'ZA'=> array('ZAF','710','ISO 3166-2:ZA'),
				'GS'=> array('SGS','239','ISO 3166-2:GS'),
				'SS'=> array('SSD','728','ISO 3166-2:SS'),
				'ES'=> array('ESP','724','ISO 3166-2:ES'),
				'LK'=> array('LKA','144','ISO 3166-2:LK'),
				'SD'=> array('SDN','729','ISO 3166-2:SD'),
				'SR'=> array('SUR','740','ISO 3166-2:SR'),
				'SJ'=> array('SJM','744','ISO 3166-2:SJ'),
				'SZ'=> array('SWZ','748','ISO 3166-2:SZ'),
				'SE'=> array('SWE','752','ISO 3166-2:SE'),
				'CH'=> array('CHE','756','ISO 3166-2:CH'),
				'SY'=> array('SYR','760','ISO 3166-2:SY'),
				'TW'=> array('TWN','158','ISO 3166-2:TW'),
				'TJ'=> array('TJK','762','ISO 3166-2:TJ'),
				'TZ'=> array('TZA','834','ISO 3166-2:TZ'),
				'TH'=> array('THA','764','ISO 3166-2:TH'),
				'TL'=> array('TLS','626','ISO 3166-2:TL'),
				'TG'=> array('TGO','768','ISO 3166-2:TG'),
				'TK'=> array('TKL','772','ISO 3166-2:TK'),
				'TO'=> array('TON','776','ISO 3166-2:TO'),
				'TT'=> array('TTO','780','ISO 3166-2:TT'),
				'TN'=> array('TUN','788','ISO 3166-2:TN'),
				'TR'=> array('TUR','792','ISO 3166-2:TR'),
				'TM'=> array('TKM','795','ISO 3166-2:TM'),
				'TC'=> array('TCA','796','ISO 3166-2:TC'),
				'TV'=> array('TUV','798','ISO 3166-2:TV'),
				'UG'=> array('UGA','800','ISO 3166-2:UG'),
				'UA'=> array('UKR','804','ISO 3166-2:UA'),
				'AE'=> array('ARE','784','ISO 3166-2:AE'),
				'GB'=> array('GBR','826','ISO 3166-2:GB'),
				'US'=> array('USA','840','ISO 3166-2:US'),
				'UM'=> array('UMI','581','ISO 3166-2:UM'),
				'UY'=> array('URY','858','ISO 3166-2:UY'),
				'UZ'=> array('UZB','860','ISO 3166-2:UZ'),
				'VU'=> array('VUT','548','ISO 3166-2:VU'),
				'VE'=> array('VEN','862','ISO 3166-2:VE'),
				'VN'=> array('VNM','704','ISO 3166-2:VN'),
				'VG'=> array('VGB','092','ISO 3166-2:VG'),
				'VI'=> array('VIR','850','ISO 3166-2:VI'),
				'WF'=> array('WLF','876','ISO 3166-2:WF'),
				'EH'=> array('ESH','732','ISO 3166-2:EH'),
				'YE'=> array('YEM','887','ISO 3166-2:YE'),
				'ZM'=> array('ZMB','894','ISO 3166-2:ZM'),
				'ZW'=> array('ZWE','716','ISO 3166-2:ZW')
			);

			if(isset($countries_3166[$order_country][1])) {
				return $countries_3166[$order_country][1];
			} else {
				return "246";  // default to Finland
			}
		}

		function get_error_message($error_code)
		{
			$error_message= array(
				'cancel-payment-rejected'       => array( 'Rejected', 'Payment was rejected buy payment acquirer due to restrictions placed on payment method or lack of funds in the corresponding account.'),
				'cancel-user-canceled'          => array( 'Buyer Canceled', 'Buyer canceled to payment process.'),
				'cancel-payment-expired'        => array( 'Expired', 'The payment process expired due to timeout or usage of browser navigation buttons.'),
				'cancel-system-error'           => array( 'Error', 'The payment process failed due to error in the payment system or payment acquirer system.'),
				'cancel-inv-payment-params'     => array( 'Error', 'The payment process failed due to payment parameters from merchant, that lead to situation that could not be fulfilled, such as an attempt to make recurring bank payment.'),
				'cancel-external-system-error'  => array( 'Error', 'They payment failed due to external system giving invalid message authentication code.'),
				'cancel-save-rej-not-enrolled'  => array( 'Rejected', 'Attempt to save a non 3d secure enrolled card, when this is not supported by acquirer agreement for this type of card.'),
				'cancel-paym-rej-not-enrolled'  => array( 'Buyer Canceled', 'Attempt to pay with non 3d secure enrolled card more that the maximum set by acquirer agreement of the merchant for that card type.'),
				'cancel-ext-address-mismatch'   => array( 'Rejected', 'External system has different address than given one.'),
				'cancel-too-many-sms-passwords' => array( 'Buyer Canceled', 'User tried to enter wrong sms password too many times.'),
				'cancel-too-many-input-retries' => array( 'Buyer Canceled', 'Page has been reloaded for too many times. Browsers have been instructed not to reload payment web pages, so if this happens payment web assumed that the request is not from browser or browser is reloading page infinitely.'),
				'cancel-bank_axess_error'       => array( 'Rejected', 'Payment rejected because of a BankAxess error code returned. (Example: J1: Bank does not support BankAxess).'),
			);

			if(isset($error_message[$error_code])) {
				return $error_message[$error_code][1];
			} else {
				return array( 'Error' , 'An error has occurred.');
			}
		}

		/**
		 * Get all pages
		 */
		function get_pages($title = false, $indent = true) {
			$wp_pages = get_pages('sort_column=menu_order');
			$page_list = array();
			if ($title) $page_list[] = $title;
			foreach ($wp_pages as $page) {
				$prefix = '';
				// show indented child pages?
				if ($indent) {
					$has_parent = $page->post_parent;
					while($has_parent) {
						$prefix .=  ' - ';
						$next_page = get_page($has_parent);
						$has_parent = $next_page->post_parent;
					}
				}
				// add to page list array array
				$page_list[$page->ID] = $prefix . $page->post_title;
			}
			return $page_list;
		}
	}
	/**
	 * Add the Gateway to WooCommerce
	 **/
	function woocommerce_add_verifone_payment_gateway($methods) {
		$methods[] = 'WC_Verifone_Payment';
		return $methods;
	}

	add_filter('woocommerce_payment_gateways', 'woocommerce_add_verifone_payment_gateway' );
}
