package com.verifone.demo_shop.util;

import java.nio.charset.Charset;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.Signature;
import java.util.Map;
import java.util.TreeMap;

import org.bouncycastle.jce.provider.BouncyCastleProvider;

/**
 * Class for utility functions.
 */
public final class VerifoneSignatureUtil {

    /**
     * Private constructor to prevent constructing of this class.
     */
    private VerifoneSignatureUtil() {}

    /**
     * Formats map of parameters to byte array for signing.
     * @param parameters the parameters to be signed.
     * @return signature.
     */
    public static byte[] formatParameters(final TreeMap<String, String> parameters) {
        final StringBuilder builder = new StringBuilder();
        for (final Map.Entry<String, String> entry : parameters.entrySet()) {
            builder.append(entry.getKey());
            builder.append('=');
            builder.append(entry.getValue());
            builder.append(';');
        }
        return builder.toString().getBytes(Charset.forName("UTF-8"));
    }

    /**
     * Verifies digital signature of specified data.
     * @param data Signed data.
     * @param signature Signature.
     * @param publicKey Public key.
     * @param encryptionAlgorithm Signature algorithm.
     * @param hashAlgorithm Hash algorithm.
     * @return true if signature is valid.
     */
    public static boolean verify(final byte[] data, final byte[] signature, final PublicKey publicKey, final String encryptionAlgorithm, final String hashAlgorithm) {
        try {
            String algorithm = null;
            if (encryptionAlgorithm.equals("RSA") && hashAlgorithm.equals("SHA-1")) {
                algorithm = "SHA1withRSA";
            } else if (encryptionAlgorithm.equals("RSA") && hashAlgorithm.equals("SHA-512")) {
                algorithm = "SHA512withRSA";
            } else {
                throw new RuntimeException("Unsupported signature algorith: " + hashAlgorithm + " with " + encryptionAlgorithm);
            }
            
            if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
                Security.addProvider(new BouncyCastleProvider());
            }

            final Signature sign = Signature.getInstance(algorithm, BouncyCastleProvider.PROVIDER_NAME);
            sign.initVerify(publicKey);
            sign.update(data);
            
            return sign.verify(signature);
        } catch (final Exception e) {
            return false;
        }
    }

    /**
     * Signs the data .
     * @param data Data to sign.
     * @param privateKey Private key.
     * @param encryptionAlgorithm Signature algorithm.
     * @param hashAlgorithm Hash algorithm.
     * @return Signature bytes.
     */
    public static byte[] sign(final byte[] data, final PrivateKey privateKey, final String encryptionAlgorithm, final String hashAlgorithm) {
        try {
            String algorithm = null;
            if (encryptionAlgorithm.equals("RSA") && hashAlgorithm.equals("SHA-1")) {
                algorithm = "SHA1withRSA";
            } else if (encryptionAlgorithm.equals("RSA") && hashAlgorithm.equals("SHA-512")) {
                algorithm = "SHA512withRSA";
            } else {
                throw new RuntimeException("Unsupported signature algorith: " + hashAlgorithm + " with " + encryptionAlgorithm);
            }
            
            if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
                Security.addProvider(new BouncyCastleProvider());
            }

            final Signature sign = Signature.getInstance(algorithm, BouncyCastleProvider.PROVIDER_NAME);
            sign.initSign(privateKey);
            sign.update(data);
            
            return sign.sign();
        } catch (final Exception e) {
            throw new RuntimeException("Error signing data.", e);
        }
    }

}
